<?php
// populate_champions_cup.php
set_time_limit(600);
include('connect.php');

/**
 * MAIN POPULATION FUNCTION
 */
function populateChampionsCupComplete($season = '2025') {
    global $db;
    
    echo "<h2>🏆 Populating Champions Cup Database (Up to GW10)</h2>";
    
    // Step 1: Clear any existing data
    clearCupData($season);
    echo "✅ Cleared existing cup data<br>";
    
    // Step 2: Populate qualification data (as if GW10 has passed)
    populateQualificationData($season);
    echo "✅ Populated qualification data<br>";
    
    // Step 3: Create groups and draw
    createCupGroups($season);
    echo "✅ Created groups and draw<br>";
    
    // Step 4: Generate group stage fixtures
    generateGroupFixtures($season, 12, 2);
    echo "✅ Generated group stage fixtures<br>";
    
    // Step 5: Populate manager histories (up to GW10)
    populateManagerHistoriesUpToGW10();
    echo "✅ Populated manager histories up to GW10<br>";
    
    // Step 6: Populate current standings (as if GW10 completed)
    populateCurrentStandings($season);
    echo "✅ Populated current standings<br>";
    
    // Step 7: Display the populated data
    displayPopulatedData($season);
    
    echo "<h3>🎉 Champions Cup database populated successfully!</h3>";
    echo "<p>The system is now ready with complete data up to GW10.</p>";
    echo "<p>You can now test qualification, group draws, and fixture generation.</p>";
}

/**
 * Clear all cup-related data
 */
function clearCupData($season) {
    global $db;
    
    $tables = [
        'cup_teams',
        'cup_groups', 
        'cup_group_teams',
        'cup_fixtures',
        'cup_goalscorers',
        'cup_standings'
    ];
    
    foreach ($tables as $table) {
        $db->query("DELETE FROM $table WHERE season_year = '" . $db->real_escape_string($season) . "'");
    }
}

/**
 * Populate qualification data as if GW10 has been completed
 */
function populateQualificationData($season) {
    global $db;
    
    // Get real teams from your database
    $teams = $db->query("SELECT id, name, division FROM teams ORDER BY division, id LIMIT 50");
    
    $qualified_teams = [];
    $team_count = 0;
    
    // Defending champion (pick a random team from last season)
    $defending_champion = $db->query("SELECT id, division FROM teams ORDER BY RAND() LIMIT 1")->fetch_assoc();
    if ($defending_champion) {
        $points = rand(900, 1100);
        $db->query("INSERT INTO cup_teams (team_id, division, qualified_by, total_points, season_year) 
                   VALUES ({$defending_champion['id']}, {$defending_champion['division']}, 
                           'Defending Champion', $points, '$season')");
        $qualified_teams[] = $defending_champion['id'];
        $team_count++;
    }
    
    // Division quotas
    $division_quotas = [1 => 12, 2 => 10, 3 => 9];
    $division_counts = [1 => 0, 2 => 0, 3 => 0];
    
    while ($team = $teams->fetch_assoc() && $team_count < 32) {
        $team_id = $team['id'];
        $division = $team['division'];
        
        // Skip if already qualified (defending champion)
        if (in_array($team_id, $qualified_teams)) continue;
        
        // Skip if division quota filled
        if ($division_counts[$division] >= $division_quotas[$division]) continue;
        
        // Generate realistic total points for GW10
        $base_points = [
            1 => rand(850, 1100),  // Division 1: higher points
            2 => rand(750, 950),   // Division 2: medium points  
            3 => rand(650, 850)    // Division 3: lower points
        ];
        
        $points = $base_points[$division];
        $qualified_by = "Division $division Top {$division_quotas[$division]}";
        
        $db->query("INSERT INTO cup_teams (team_id, division, qualified_by, total_points, season_year) 
                   VALUES ($team_id, $division, '$qualified_by', $points, '$season')");
        
        $qualified_teams[] = $team_id;
        $division_counts[$division]++;
        $team_count++;
    }
    
    return $qualified_teams;
}

/**
 * Populate manager histories with realistic data up to GW10
 */
function populateManagerHistoriesUpToGW10() {
    global $db;
    
    // Get all managers from qualified teams
    $managers = [];
    $result = $db->query("SELECT DISTINCT t.manager1, t.manager2, t.manager3, t.manager4, t.manager5 
                         FROM teams t 
                         JOIN cup_teams ct ON t.id = ct.team_id");
    
    while ($row = $result->fetch_assoc()) {
        for ($i = 1; $i <= 5; $i++) {
            $manager_id = $row["manager$i"];
            if ($manager_id && !in_array($manager_id, $managers)) {
                $managers[] = $manager_id;
            }
        }
    }
    
    // Clear existing test data
    $db->query("DELETE FROM manager_history WHERE event <= 10");
    
    // Populate manager history for GW1-10
    foreach ($managers as $manager_id) {
        $total_points = 0;
        
        for ($gw = 1; $gw <= 10; $gw++) {
            // Generate realistic FPL points per GW
            $points = rand(40, 100);
            $transfer_cost = rand(0, 12);
            $net_points = $points - $transfer_cost;
            $bench_points = rand(5, 25);
            $vice_captain_points = rand(2, 15);
            $team_goals = rand(0, 4);
            
            $db->query("INSERT INTO manager_history 
                       (manager_id, event, points, transfer_cost, net_points, bench_points, vice_captain_points, team_goals) 
                       VALUES ($manager_id, $gw, $points, $transfer_cost, $net_points, $bench_points, $vice_captain_points, $team_goals)");
            
            $total_points += $net_points;
        }
    }
    
    echo "Populated manager histories for " . count($managers) . " managers (GW1-10)<br>";
}

/**
 * Populate current standings as if we're at GW10
 */
function populateCurrentStandings($season) {
    global $db;
    
    // This would normally be calculated from actual results
    // For simulation, we'll create realistic looking standings
    
    $teams = $db->query("SELECT team_id FROM cup_teams WHERE season_year = '$season'");
    
    while ($team = $teams->fetch_assoc()) {
        $team_id = $team['team_id'];
        
        // Get team's division for realistic stats
        $div_result = $db->query("SELECT division FROM teams WHERE id = $team_id");
        $division = $div_result->fetch_assoc()['division'];
        
        // Generate realistic stats based on division
        $stats = generateRealisticTeamStats($division);
        
        $db->query("INSERT INTO team_standings 
                   (team_id, division, played, won, drawn, lost, goals_for, goals_against, points, season_year) 
                   VALUES ($team_id, $division, {$stats['played']}, {$stats['won']}, {$stats['drawn']}, 
                           {$stats['lost']}, {$stats['gf']}, {$stats['ga']}, {$stats['points']}, '$season')");
    }
}

/**
 * Generate realistic team statistics based on division
 */
function generateRealisticTeamStats($division) {
    $base_stats = [
        1 => [ // Division 1 - top performers
            'played' => 10,
            'won' => rand(6, 8),
            'points' => rand(18, 24),
            'gf' => rand(15, 25),
            'ga' => rand(5, 15)
        ],
        2 => [ // Division 2 - medium performers
            'played' => 10, 
            'won' => rand(4, 6),
            'points' => rand(12, 18),
            'gf' => rand(10, 20),
            'ga' => rand(10, 20)
        ],
        3 => [ // Division 3 - lower performers
            'played' => 10,
            'won' => rand(2, 5), 
            'points' => rand(6, 15),
            'gf' => rand(5, 15),
            'ga' => rand(15, 25)
        ]
    ];
    
    $stats = $base_stats[$division];
    $stats['drawn'] = rand(1, 3);
    $stats['lost'] = $stats['played'] - $stats['won'] - $stats['drawn'];
    $stats['gd'] = $stats['gf'] - $stats['ga'];
    
    return $stats;
}

/**
 * Display all populated data for verification
 */
function displayPopulatedData($season) {
    global $db;
    
    echo "<h3>📊 Populated Data Verification</h3>";
    
    // Show qualified teams
    echo "<h4>Qualified Teams (32)</h4>";
    $qualified = $db->query("SELECT ct.*, t.name as team_name 
                            FROM cup_teams ct 
                            JOIN teams t ON ct.team_id = t.id 
                            WHERE ct.season_year = '$season' 
                            ORDER BY ct.division, ct.total_points DESC");
    
    echo "<table border='1' cellpadding='5' style='margin: 10px;'>";
    echo "<tr><th>Team</th><th>Division</th><th>Qualified By</th><th>Total Points</th></tr>";
    
    while ($team = $qualified->fetch_assoc()) {
        echo "<tr>";
        echo "<td>{$team['team_name']}</td>";
        echo "<td>{$team['division']}</td>";
        echo "<td>{$team['qualified_by']}</td>";
        echo "<td>{$team['total_points']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Show groups
    echo "<h4>Group Draw</h4>";
    $groups = $db->query("SELECT cg.group_name, t.name as team_name, cgt.group_seed 
                         FROM cup_groups cg 
                         JOIN cup_group_teams cgt ON cg.id = cgt.group_id 
                         JOIN teams t ON cgt.team_id = t.id 
                         WHERE cg.season_year = '$season' 
                         ORDER BY cg.group_name, cgt.group_seed");
    
    $current_group = '';
    while ($row = $groups->fetch_assoc()) {
        if ($current_group != $row['group_name']) {
            if ($current_group != '') echo "</table><br>";
            echo "<h5>Group {$row['group_name']}</h5>";
            echo "<table border='1' cellpadding='5' style='margin: 10px;'>";
            echo "<tr><th>Seed</th><th>Team</th></tr>";
            $current_group = $row['group_name'];
        }
        echo "<tr>";
        echo "<td>{$row['group_seed']}</td>";
        echo "<td>{$row['team_name']}</td>";
        echo "</tr>";
    }
    if ($current_group != '') echo "</table>";
    
    // Show fixtures
    echo "<h4>Group Stage Fixtures</h4>";
    $fixtures = $db->query("SELECT cf.gameweek, t1.name as home_team, t2.name as away_team, cg.group_name 
                           FROM cup_fixtures cf 
                           JOIN teams t1 ON cf.home_team_id = t1.id 
                           JOIN teams t2 ON cf.away_team_id = t2.id 
                           JOIN cup_group_teams cgt ON cf.home_team_id = cgt.team_id 
                           JOIN cup_groups cg ON cgt.group_id = cg.id 
                           WHERE cf.season_year = '$season' AND cf.stage = 'Group' 
                           ORDER BY cf.gameweek, cg.group_name");
    
    $current_gw = '';
    while ($fixture = $fixtures->fetch_assoc()) {
        if ($current_gw != $fixture['gameweek']) {
            if ($current_gw != '') echo "</table><br>";
            echo "<h5>Gameweek {$fixture['gameweek']}</h5>";
            echo "<table border='1' cellpadding='5' style='margin: 10px;'>";
            echo "<tr><th>Group</th><th>Home Team</th><th>Away Team</th></tr>";
            $current_gw = $fixture['gameweek'];
        }
        echo "<tr>";
        echo "<td>{$fixture['group_name']}</td>";
        echo "<td>{$fixture['home_team']}</td>";
        echo "<td>{$fixture['away_team']}</td>";
        echo "</tr>";
    }
    if ($current_gw != '') echo "</table>";
    
    // Show manager history sample
    echo "<h4>Manager History Sample (First 5 managers)</h4>";
    $managers = $db->query("SELECT DISTINCT manager_id FROM manager_history WHERE event <= 10 LIMIT 5");
    
    while ($manager = $managers->fetch_assoc()) {
        $manager_id = $manager['manager_id'];
        $history = $db->query("SELECT event, net_points FROM manager_history 
                              WHERE manager_id = $manager_id AND event <= 10 
                              ORDER BY event");
        
        echo "<p>Manager $manager_id: ";
        $points = [];
        while ($h = $history->fetch_assoc()) {
            $points[] = "GW{$h['event']}: {$h['net_points']}";
        }
        echo implode(", ", $points) . "</p>";
    }
}

// Run the population
populateChampionsCupComplete('2025');

echo "<hr>";
echo "<h3>🚀 Next Steps:</h3>";
echo "<ol>";
echo "<li><strong>Test Qualification:</strong> Verify 32 teams are properly qualified</li>";
echo "<li><strong>Test Group Draw:</strong> Check 8 groups with 4 teams each</li>";
echo "<li><strong>Test Fixture Generation:</strong> Verify 96 group fixtures scheduled</li>";
echo "<li><strong>Test Results Simulation:</strong> Run simulation for GW12-22</li>";
echo "<li><strong>Test Knockout Generation:</strong> Verify automatic knockout creation</li>";
echo "</ol>";

echo "<p><a href='test_cup_simulation.php'>→ Proceed to Simulation Test</a></p>";
?>