<?php
// set_time_limit(600);
include('../connect.php');



function getOverallTeamTotals($upToGw)
{
    global $db;
    // Returns associative: team_id => total_net_points
    $totals = [];

    // Efficient single query: join teams -> manager ids -> manager_history
    // But simplest: fetch teams and their managers then sum via manager_history grouping.
    $teamsRes = $db->query("SELECT id, manager1, manager2, manager3, manager4, manager5 FROM teams");
    $teams = [];
    while ($t = mysqli_fetch_assoc($teamsRes)) {
        $teams[$t['id']] = [
            'managers' => array_filter([$t['manager1'], $t['manager2'], $t['manager3'], $t['manager4'], $t['manager5']])
        ];
        $totals[$t['id']] = 0;
    }

    // Pull all manager_history up to $upToGw once
    $mhRes = $db->query("SELECT manager_id, event, net_points FROM manager_history WHERE event <= " . intval($upToGw));
    $mgrPoints = [];
    while ($r = mysqli_fetch_assoc($mhRes)) {
        $mgrPoints[$r['manager_id']][$r['event']] = intval($r['net_points']);
    }

    // Sum per team
    foreach ($teams as $teamId => $t) {
        $sum = 0;
        foreach ($t['managers'] as $m) {
            if (isset($mgrPoints[$m])) {
                foreach ($mgrPoints[$m] as $gw => $p) $sum += $p;
            }
        }
        $totals[$teamId] = $sum;
    }

    return $totals; // team_id => points
}


function generateCupQualifiers($season = '2025')
{
    global $db;
    $upToGw = 10;
    $totals = getOverallTeamTotals($upToGw);

    // defending champion id — fetch from your history table
    $defenderRes = $db->query("SELECT winner_team_id FROM cup_history ORDER BY id DESC LIMIT 1");
    $defender = mysqli_fetch_assoc($defenderRes)['winner_team_id'] ?? null;
    $defenderDiv = getTeamDivision($defender);

    // clear previous cup_teams for season (if re-run)
    $db->query("DELETE FROM cup_teams WHERE season_year='" . $db->real_escape_string($season) . "'");

    // Insert defending champion
    if ($defender) {
        $db->query("INSERT INTO cup_teams (team_id, division, qualifying_rank, qualified_at_gw, seed, season_year) VALUES ($defender, $defenderDiv, 0, $upToGw, 0, '" . $db->real_escape_string($season) . "')");
    }

    // Now pick top teams per division
    // assume divisions stored in teams.division (1,2,3)
    $limits = [1 => 12, 2 => 10, 3 => 9];
    foreach ($limits as $div => $limit) {
        // get all teams in division and their totals
        $teamRes = $db->query("SELECT id FROM teams WHERE division = $div AND id != $defender");
        $divTeams = [];
        while ($r = mysqli_fetch_assoc($teamRes)) $divTeams[] = $r['id'];

        // build sortable array of (team => points)
        $arr = [];
        foreach ($divTeams as $tid) {
            $arr[$tid] = $totals[$tid] ?? 0;
        }
        arsort($arr); // desc by points

        $rank = 1;
        foreach ($arr as $tid => $pts) {
            if ($rank > $limit) break;
            $seed = $rank + ($div == 1 ? 0 : ($div == 2 ? 12 : 22)); // optional seeding offset
            $db->query("INSERT INTO cup_teams (team_id, division, qualifying_rank, qualified_at_gw, seed, season_year) VALUES ($tid, $div, $rank, $upToGw, $seed, '" . $db->real_escape_string($season) . "')");
            $rank++;
        }
    }
}


function createCupGroups($season = '2025')
{
    global $db;
    // read qualified teams and their seed
    $q = $db->query("SELECT team_id, seed FROM cup_teams WHERE season_year='" . $db->real_escape_string($season) . "' ORDER BY seed ASC");
    $teams = [];
    while ($r = mysqli_fetch_assoc($q)) $teams[] = $r;

    if (count($teams) !== 32) {
        throw new Exception("Qualification not 32 teams: " . count($teams));
    }

    // Build 4 pots by seed order (1..8,9..16 etc.)
    $pots = array_chunk($teams, 8); // pots[0]=top8, pots[1]=next8, etc.

    // clear groups for season
    $db->query("DELETE FROM cup_groups WHERE season_year='" . $db->real_escape_string($season) . "'");
    $db->query("DELETE FROM cup_group_teams WHERE id IN (SELECT cgt.id FROM cup_group_teams cgt JOIN cup_groups cg ON cgt.group_id = cg.id WHERE cg.season_year = '" . $db->real_escape_string($season) . "')");

    $groupNames = range('A', 'H');
    // create empty groups
    foreach ($groupNames as $g) {
        $db->query("INSERT INTO cup_groups (group_name, season_year) VALUES ('{$g}', '" . $db->real_escape_string($season) . "')");
        $gid = $db->insert_id;
        // For each pot pick a team at random without duplication
        for ($pot = 0; $pot < 4; $pot++) {
            // pick a random entry from pots[$pot]
            $idx = array_rand($pots[$pot]);
            $team = $pots[$pot][$idx];
            // remove chosen
            array_splice($pots[$pot], $idx, 1);
            // record in group_team with seed 1..4
            $db->query("INSERT INTO cup_group_teams (group_id, team_id, group_seed) VALUES ($gid, {$team['team_id']}, " . ($pot + 1) . ")");
        }
    }
}


function generateGroupFixtures($season = '2025', $start_gw = 12, $interval = 2)
{
    global $db;

    // Clear existing fixtures for season/stage
    $db->query("DELETE FROM cup_fixtures WHERE season_year='" . $db->real_escape_string($season) . "' AND stage='Group'");

    // Get all groups
    $groups = $db->query("SELECT id, group_name FROM cup_groups WHERE season_year='" . $db->real_escape_string($season) . "'");
    $all_groups = [];
    while ($g = $groups->fetch_assoc()) {
        $all_groups[] = $g;
    }

    // For each group, generate a proper round-robin schedule
    $group_schedules = [];

    foreach ($all_groups as $group) {
        // Fetch teams in group
        $res = $db->query("SELECT team_id FROM cup_group_teams WHERE group_id={$group['id']} ORDER BY group_seed ASC");
        $teams = [];
        while ($t = $res->fetch_assoc()) {
            $teams[] = $t['team_id'];
        }

        // Generate round-robin schedule (3 matchdays for first leg)
        $schedule = generateRoundRobinSchedule($teams);
        $group_schedules[$group['id']] = $schedule;
    }

    // Schedule matchdays (6 total - 3 first leg, 3 second leg)
    $current_gw = $start_gw;

    // First leg: Matchdays 1-3
    for ($matchday = 1; $matchday <= 3; $matchday++) {
        foreach ($all_groups as $group) {
            $group_id = $group['id'];
            if (isset($group_schedules[$group_id][$matchday - 1])) {
                // Each matchday has multiple fixtures, so we need to loop through them
                $matchday_fixtures = $group_schedules[$group_id][$matchday - 1];
                foreach ($matchday_fixtures as $fixture) {
                    if (isset($fixture['home']) && isset($fixture['away'])) {
                        $home = $fixture['home'];
                        $away = $fixture['away'];

                        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, matchday, status) 
                                   VALUES ('Group', '" . $db->real_escape_string($season) . "', $current_gw, $home, $away, 1, $matchday, 'Pending')");
                    }
                }
            }
        }
        $current_gw += $interval;
    }

    // Second leg: Matchdays 4-6 (reverse fixtures)
    for ($matchday = 4; $matchday <= 6; $matchday++) {
        foreach ($all_groups as $group) {
            $group_id = $group['id'];
            $first_leg_matchday = $matchday - 3; // Corresponding first leg matchday
            if (isset($group_schedules[$group_id][$first_leg_matchday - 1])) {
                $matchday_fixtures = $group_schedules[$group_id][$first_leg_matchday - 1];
                foreach ($matchday_fixtures as $fixture) {
                    if (isset($fixture['home']) && isset($fixture['away'])) {
                        // Reverse home/away for second leg
                        $home = $fixture['away'];
                        $away = $fixture['home'];

                        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, matchday, status) 
                                   VALUES ('Group', '" . $db->real_escape_string($season) . "', $current_gw, $home, $away, 2, $matchday, 'Pending')");
                    }
                }
            }
        }
        $current_gw += $interval;
    }

    return true;
}

/**
 * Generate round-robin schedule for 4 teams (3 matchdays)
 * Returns array of matchdays, each with fixtures
 */
function generateRoundRobinSchedule($teams)
{
    if (count($teams) != 4) {
        throw new Exception("Round-robin schedule requires exactly 4 teams");
    }

    // Standard round-robin schedule for 4 teams
    $schedule = [
        [ // Matchday 1
            ['home' => $teams[0], 'away' => $teams[1]],
            ['home' => $teams[2], 'away' => $teams[3]]
        ],
        [ // Matchday 2  
            ['home' => $teams[0], 'away' => $teams[2]],
            ['home' => $teams[1], 'away' => $teams[3]]
        ],
        [ // Matchday 3
            ['home' => $teams[0], 'away' => $teams[3]],
            ['home' => $teams[1], 'away' => $teams[2]]
        ]
    ];

    return $schedule;
}


function checkAndGenerateKnockouts($season = '2025')
{
    global $db;
    // Are all Group stage fixtures Closed?
    $left = $db->query("SELECT COUNT(*) AS cnt FROM cup_fixtures WHERE season_year='" . $db->real_escape_string($season) . "' AND stage='Group' AND status != 'Closed'")->fetch_assoc()['cnt'];
    if ($left > 0) return false; // still ongoing

    // compute group standings
    $groups = $db->query("SELECT id FROM cup_groups WHERE season_year='" . $db->real_escape_string($season) . "'");
    $r16_teams = [];

    while ($g = $groups->fetch_assoc()) {
        $gid = $g['id'];
        // initialize stats
        $stats = []; // team_id => [points, gf, ga, gd]
        $res = $db->query("SELECT team_id FROM cup_group_teams WHERE group_id=$gid");
        while ($t = $res->fetch_assoc()) $stats[$t['team_id']] = ['points' => 0, 'gf' => 0, 'ga' => 0];

        // get fixtures for this group
        $fx = $db->query("SELECT * FROM cup_fixtures WHERE stage='Group' AND season_year='" . $db->real_escape_string($season) . "' AND (home_team_id IN (SELECT team_id FROM cup_group_teams WHERE group_id=$gid) AND away_team_id IN (SELECT team_id FROM cup_group_teams WHERE group_id=$gid))");
        while ($f = $fx->fetch_assoc()) {
            if ($f['home_goals'] === null || $f['away_goals'] === null) continue;
            $home = $f['home_team_id'];
            $away = $f['away_team_id'];
            $hg = intval($f['home_goals']);
            $ag = intval($f['away_goals']);
            $stats[$home]['gf'] += $hg;
            $stats[$home]['ga'] += $ag;
            $stats[$away]['gf'] += $ag;
            $stats[$away]['ga'] += $hg;
            if ($hg > $ag) {
                $stats[$home]['points'] += 3;
            } elseif ($hg < $ag) {
                $stats[$away]['points'] += 3;
            } else {
                $stats[$home]['points'] += 1;
                $stats[$away]['points'] += 1;
            }
        }

        // compute GD and sort
        foreach ($stats as $tid => &$s) $s['gd'] = $s['gf'] - $s['ga'];
        uasort($stats, function ($a, $b) {
            if ($a['points'] !== $b['points']) return $b['points'] - $a['points'];
            if ($a['gd'] !== $b['gd']) return $b['gd'] - $a['gd'];
            if ($a['gf'] !== $b['gf']) return $b['gf'] - $a['gf'];
            return 0;
        });

        // pick top 2
        $group_order = array_keys($stats);
        $top2 = array_slice($group_order, 0, 2);
        $r16_teams = array_merge($r16_teams, $top2);
    }

    // Now create Round of 16 pairings (simple bracket: A1 vs B2 etc.)
    // Example mapping (A1 v B2), (C1 v D2), (E1 v F2), (G1 v H2), (B1 v A2), (D1 v C2), (F1 v E2), (H1 v G2)
    // We need to map groups to their A1, B2 etc. Build map:
    $groupTeams = [];
    $grRes = $db->query("SELECT id, group_name FROM cup_groups WHERE season_year='" . $db->real_escape_string($season) . "'");
    while ($g = $grRes->fetch_assoc()) {
        // obtain ordered teams (same code as above can be refactored into a function)
        // For brevity assume $orderedTeams = computeOrderedTeamsForGroup($g['id']);
        $orderedTeams = computeOrderedTeamsForGroup($g['id'], $season); // you implement
        $groupTeams[$g['group_name']] = $orderedTeams; // array of team_ids, 0-based [0]=1st
    }

    // create R16 pairings (example)
    $r16_matches = [
        [$groupTeams['A'][0], $groupTeams['B'][1]],
        [$groupTeams['C'][0], $groupTeams['D'][1]],
        [$groupTeams['E'][0], $groupTeams['F'][1]],
        [$groupTeams['G'][0], $groupTeams['H'][1]],
        [$groupTeams['B'][0], $groupTeams['A'][1]],
        [$groupTeams['D'][0], $groupTeams['C'][1]],
        [$groupTeams['F'][0], $groupTeams['E'][1]],
        [$groupTeams['H'][0], $groupTeams['G'][1]]
    ];

    // schedule R16 fixtures: two legs. Choose GWs: e.g. start at GW( last group gw + 2) and every 2 gws
    $start_gw = determineR16StartGW($season);
    $interval = 2;
    $tieIndex = 1;

    $check = $db->query("SELECT * FROM cup_fixtures WHERE stage = 'R16' AND season_year = '" . $db->real_escape_string($season) . "'");
    if (mysqli_num_rows($check) >= 1) {
        $db->query("DELETE FROM cup_fixtures WHERE stage = 'R16' AND season_year = '" . $db->real_escape_string($season) . "'");
    }


    foreach ($r16_matches as $pair) {

        // leg 1
        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) VALUES ('R16','" . $db->real_escape_string($season) . "',$start_gw, {$pair[0]}, {$pair[1]}, 1, 'R16-{$tieIndex}', 'Pending')");
        // leg 2
        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) VALUES ('R16','" . $db->real_escape_string($season) . "'," . ($start_gw + $interval) . ", {$pair[1]}, {$pair[0]}, 2, 'R16-{$tieIndex}', 'Pending')");

        $tieIndex++;
    }

    return true;
}


/**
 * Compute ordered teams for group (referenced in checkAndGenerateKnockouts)
 */
function computeOrderedTeamsForGroup($group_id, $season)
{
    global $db;
    // Get all fixtures for teams in this group
    $teamRes = $db->query("SELECT team_id FROM cup_group_teams WHERE group_id = $group_id");
    $teamIds = [];
    while ($t = mysqli_fetch_assoc($teamRes)) $teamIds[] = $t['team_id'];
    $teamIdsStr = implode(',', $teamIds);

    // Initialize stats
    $stats = [];
    foreach ($teamIds as $tid) {
        $stats[$tid] = ['points' => 0, 'gf' => 0, 'ga' => 0, 'gd' => 0];
    }

    // Calculate from fixtures
    $fx = $db->query("SELECT * FROM cup_fixtures WHERE stage = 'Group' AND season_year = '" . $db->real_escape_string($season) . "' AND home_team_id IN ($teamIdsStr) AND away_team_id IN ($teamIdsStr)");

    while ($f = mysqli_fetch_assoc($fx)) {
        if ($f['home_goals'] === null || $f['away_goals'] === null) continue;

        $home = $f['home_team_id'];
        $away = $f['away_team_id'];
        $hg = intval($f['home_goals']);
        $ag = intval($f['away_goals']);

        $stats[$home]['gf'] += $hg;
        $stats[$home]['ga'] += $ag;
        $stats[$away]['gf'] += $ag;
        $stats[$away]['ga'] += $hg;

        if ($hg > $ag) {
            $stats[$home]['points'] += 3;
        } elseif ($hg < $ag) {
            $stats[$away]['points'] += 3;
        } else {
            $stats[$home]['points'] += 1;
            $stats[$away]['points'] += 1;
        }
    }

    // Compute GD and sort
    foreach ($stats as $tid => &$s) {
        $s['gd'] = $s['gf'] - $s['ga'];
    }

    uasort($stats, function ($a, $b) {
        if ($a['points'] !== $b['points']) return $b['points'] - $a['points'];
        if ($a['gd'] !== $b['gd']) return $b['gd'] - $a['gd'];
        if ($a['gf'] !== $b['gf']) return $b['gf'] - $a['gf'];
        return 0;
    });

    return array_keys($stats); // Returns team IDs in order
}

/**
 * Determine R16 start GW (referenced in checkAndGenerateKnockouts)
 */
function determineR16StartGW($season)
{
    global $db;
    // Find the last group stage GW and add 2
    $result = $db->query("SELECT MAX(gameweek) as last_gw FROM cup_fixtures WHERE stage = 'Group' AND season_year = '" . $db->real_escape_string($season) . "'");
    $row = $result->fetch_assoc();
    return $row['last_gw'] + 2;
}














/**
 * Main workflow that integrates with your existing system
 */
function runIntegratedChampionsCupWorkflow($current_gw, $season = '2025')
{
    global $db;

    if ($current_gw < 10) {
        echo 'not yet';
        return;
    }

    // QUALIFICATION PHASE (GW10)
    if ($current_gw == 10) {
        generateCupQualifiers($season);
        createCupGroups($season);
        generateGroupFixtures($season, 12, 2);
        echo "Champions Cup: Qualification completed for $season\n";
    }

    // CUP MATCHWEEKS (every 2 weeks starting GW12)
    $cup_gameweeks = [12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 38];

    if (in_array($current_gw, $cup_gameweeks)) {
        // Update cup results using your existing points system
        updateCupResultsWithPoints($current_gw, $season);

        // Update group standings after each matchday
        if ($current_gw <= 22) { // Group stage
            updateAllCupGroupStandings($season);
        }

        // Check if group stage is complete and generate knockouts
        if ($current_gw == 22 && isCupGroupStageComplete($season)) {
            checkAndGenerateKnockouts($season);
        }

        // Update knockout results
        if ($current_gw > 22) {
            updateCupKnockoutResults($current_gw, $season);
        }
    }
}
runIntegratedChampionsCupWorkflow(36);

/**
 * Check if all group stage fixtures are complete
 */
function isCupGroupStageComplete($season = '2025')
{
    global $db;

    $result = $db->query("SELECT COUNT(*) as incomplete 
                         FROM cup_fixtures 
                         WHERE stage = 'Group' 
                         AND season_year = '" . $db->real_escape_string($season) . "' 
                         AND status != 'Closed'");
    $row = $result->fetch_assoc();

    return $row['incomplete'] == 0;
}

/**
 * Update standings for all groups
 */
function updateAllCupGroupStandings($season = '2025')
{
    global $db;

    $groups = $db->query("SELECT id FROM cup_groups 
                         WHERE season_year = '" . $db->real_escape_string($season) . "'");

    while ($group = $groups->fetch_assoc()) {
        $standings = computeCupGroupStandings($group['id'], $season);
        // You could store these in a cup_standings table if needed
    }
}

/**
 * Get current gameweek from your system
 */
function getCurrentGameweek()
{
    // Use your existing implementation
    global $db;
    $result = $db->query("SELECT MAX(gameweek) as current_gw FROM fixtures WHERE deadline < NOW()");
    $row = $result->fetch_assoc();
    return $row['current_gw'] ?? 1;
}

/**
 * Update cup results using your existing FPL points system and goal rules
 */
function updateCupResultsWithPoints($gameweek, $season = 2025)
{
    global $db;

    // Get all pending cup fixtures for this GW
    $fixtures = $db->query("SELECT * FROM cup_fixtures 
                           WHERE gameweek = $gameweek 
                           AND season_year = '" . $db->real_escape_string($season) . "' 
                           AND status != 'Closed'");

    $updated = 0;

    while ($fixture = mysqli_fetch_assoc($fixtures)) {
        // Get team points using your existing system
        $home_points = getCupTeamPoints($fixture['home_team_id'], $gameweek);
        $away_points = getCupTeamPoints($fixture['away_team_id'], $gameweek);


        // Apply your existing goal rules
        $goals = applyGoalRules($home_points, $away_points);

        // Update fixture with goals and status

        if ($fixture['stage'] != 'Group') {
            $db->query("UPDATE cup_fixtures SET home_goals = {$goals['home_goals']},  away_goals = {$goals['away_goals']}, 
                       status = 'Pending' 
                   WHERE id = {$fixture['id']}");
        } else {
            $db->query("UPDATE cup_fixtures SET home_goals = {$goals['home_goals']},  away_goals = {$goals['away_goals']}, 
                       status = 'Closed' 
                   WHERE id = {$fixture['id']}");
        }

        // Update goalscorers for cup (if applicable)
        // updateCupGoalscorers($fixture, $goals, $gameweek, 2025);

        // Skip the rest of processing for draws, but continue with other fixtures
        if ($goals['home_goals'] == $goals['away_goals']) {
            $updated++;
            continue; // This continues to the next fixture instead of exiting the function
        }

        $winning_team_id = $goals['home_goals'] > $goals['away_goals'] ?
            $fixture['home_team_id'] : $fixture['away_team_id'];

        $winning_managers = getTeamManagers($winning_team_id);

        if (is_string($winning_managers)) {
            $try = json_decode($winning_managers, true);
            $winning_managers = is_array($try) ? $try : [];
        }

        // Find top scoring manager (using your existing logic)
        $top_manager = null;
        $top_points = -INF;
        $top_vc = -INF;
        $top_bench = -INF;
        $top_goals = -INF;


        foreach ($winning_managers as $m) {
            $result = $db->query("SELECT * FROM manager_history 
                             WHERE manager_id = {$m} AND event = $gameweek");
            $h = $result->fetch_assoc();


            if (!$h) continue;

            $net = $h['net_points'];
            $vc = $h['vice_captain_points'];
            $bench = $h['bench_points'];
            $tgoals = $h['team_goals'];

            if (
                $net > $top_points ||
                ($net == $top_points && (
                    $vc > $top_vc ||
                    ($vc == $top_vc && (
                        $bench > $top_bench ||
                        ($bench == $top_bench && $tgoals > $top_goals)
                    ))
                ))
            ) {
                $top_points = $net;
                $top_vc = $vc;
                $top_bench = $bench;
                $top_goals = $tgoals;
                $top_manager = $m;
            }
        }

        if ($top_manager) {
            // Insert into cup goalscorers table (create this table if needed)
            $db->query("INSERT INTO cup_goalscorers 
                   (gameweek, fixture_id, team_id, manager_id, points, season_year) 
                   VALUES ('$gameweek', '{$fixture['id']}', '$winning_team_id', 
                           '$top_manager', '$top_points', 
                           '" . $db->real_escape_string($season) . "')");
        }

        $updated++;
    }

    echo ("Cup: Updated $updated fixtures for GW$gameweek");
    return $updated;
}



function applyGoalRules($homePoints, $awayPoints)
{
    global $db;
    // load goal rules
    $goalRules = [];
    $qr = $db->query("SELECT id, min_points, max_points, goals FROM goal_rules ORDER BY min_points ASC");
    while ($r = mysqli_fetch_assoc($qr)) {
        $goalRules[] = [
            'id' => intval($r['id']),
            'min' => intval($r['min_points']),
            'max' => intval($r['max_points']),
            'goals' => intval($r['goals'])
        ];
    }

    if (empty($goalRules)) {
        return ['error' => 'No goal rules found'];
    }

    // helper: find band by points
    $findBand = function ($points) use ($goalRules) {
        foreach ($goalRules as $band) {
            if ($points >= $band['min'] && $points <= $band['max']) {
                return $band;
            }
        }
        return null;
    };

    $homeBand = $findBand($homePoints);
    $awayBand = $findBand($awayPoints);

    if (!$homeBand || !$awayBand) {
        return ['error' => 'Points do not fall into any goal band', 'home_band' => $homeBand, 'away_band' => $awayBand];
    }

    // load settings in one query
    $settings = [
        'same_band_margin' => 5,
        'diff_band_margin' => 5,
        'extra_goal_enabled' => 0,
        'extra_goal_margin' => 10
    ];

    $query = $db->query("SELECT diff_band_margin, same_band_margin, extra_goal_enabled, extra_goal_margin FROM settings ORDER BY id ASC");

    $rules = [];
    $settings = [];
    while ($row = mysqli_fetch_assoc($query)) {
        $rules[] = $row;
    }

    $settings = $rules[0];

    $homeGoals = $homeBand['goals'];
    $awayGoals = $awayBand['goals'];
    $margin = abs($homePoints - $awayPoints);
    $explanation = '';

    // same band logic
    if ($homeBand['id'] === $awayBand['id']) {
        if ($margin > intval($settings['same_band_margin'])) {
            if ($homePoints > $awayPoints) {
                $homeGoals += 1;
                $explanation = "Same band, margin > {$settings['same_band_margin']} → +1 to Home";
            } else {
                $awayGoals += 1;
                $explanation = "Same band, margin > {$settings['same_band_margin']} → +1 to Away";
            }
        } else {
            $explanation = "Same band, margin ≤ {$settings['same_band_margin']} → Draw";
        }
    } else {
        // different bands
        if ($margin <= intval($settings['diff_band_margin'])) {
            $drawGoals = min($homeGoals, $awayGoals);
            $homeGoals = $drawGoals;
            $awayGoals = $drawGoals;
            $explanation = "Different bands, margin ≤ {$settings['diff_band_margin']} → Forced draw at lower band goals";
        } else {
            $explanation = "Different bands, margin > {$settings['diff_band_margin']} → Goals by band";
        }
    }

    // extra goal rule per your request:
    // if extra_goal_enabled == 0 then add one goal to higher points side when margin <= extra_goal_margin
    if (intval($settings['extra_goal_enabled']) === 0) {
        $egm = intval($settings['extra_goal_margin']);
        if ($margin <= $egm && $margin > 0) {
            if ($homePoints > $awayPoints) {
                $homeGoals += 1;
                $explanation .= " | Extra rule: +1 to Home (margin ≤ {$egm})";
            } elseif ($awayPoints > $homePoints) {
                $awayGoals += 1;
                $explanation .= " | Extra rule: +1 to Away (margin ≤ {$egm})";
            }
        }
    }

    return [
        'home_points' => $homePoints,
        'away_points' => $awayPoints,
        'home_goals' => intval($homeGoals),
        'away_goals' => intval($awayGoals),
        'home_band' => $homeBand,
        'away_band' => $awayBand,
        'margin' => $margin,
        'explanation' => $explanation
    ];
}


/**
 * Get team points for cup using your existing manager system
 */
function getCupTeamPoints($team_id, $gameweek)
{
    global $db;
    // Use your existing function to get team managers
    $managers = getTeamManagers($team_id);

    if (is_string($managers)) {
        $try = json_decode($managers, true);
        $managers = is_array($try) ? $try : [];
    }

    // Use your existing sumManagersPoints function
    // We need to get manager histories - we'll use the cached approach from your updateAllResults
    // $managerIds = array_column($managers, 'manager_id');
    // echo json_encode($managerIds);

    // Get manager histories from database (using your cached approach)
    $total_points = 0;
    foreach ($managers as $managerId) {
        $result = $db->query("SELECT net_points FROM manager_history 
                             WHERE manager_id = $managerId AND event = $gameweek");
        if ($row = $result->fetch_assoc()) {
            $total_points += $row['net_points'];
        }
    }

    return $total_points;
}


/**
 * Update cup goalscorers using your existing logic
 */
function updateCupGoalscorers($fixture, $goals, $gameweek, $season)
{
    global $db;

    // Only update if there's a winner (not a draw)
    if ($goals['home_goals'] == $goals['away_goals']) return;

    $winning_team_id = $goals['home_goals'] > $goals['away_goals'] ?
        $fixture['home_team_id'] : $fixture['away_team_id'];

    $winning_managers = getTeamManagers($winning_team_id);

    if (is_string($winning_managers)) {
        $try = json_decode($winning_managers, true);
        $winning_managers = is_array($try) ? $try : [];
    }

    // Find top scoring manager (using your existing logic)
    $top_manager = null;
    $top_points = -INF;
    $top_vc = -INF;
    $top_bench = -INF;
    $top_goals = -INF;

    foreach ($winning_managers as $m) {
        $result = $db->query("SELECT * FROM manager_history 
                             WHERE manager_id = {$m['manager_id']} AND event = $gameweek");
        $h = $result->fetch_assoc();

        if (!$h) continue;

        $net = $h['net_points'];
        $vc = $h['vice_captain_points'];
        $bench = $h['bench_points'];
        $tgoals = $h['team_goals'];

        if (
            $net > $top_points ||
            ($net == $top_points && (
                $vc > $top_vc ||
                ($vc == $top_vc && (
                    $bench > $top_bench ||
                    ($bench == $top_bench && $tgoals > $top_goals)
                ))
            ))
        ) {
            $top_points = $net;
            $top_vc = $vc;
            $top_bench = $bench;
            $top_goals = $tgoals;
            $top_manager = $m['manager_id'];
        }
    }

    if ($top_manager) {
        // Insert into cup goalscorers table (create this table if needed)
        $db->query("INSERT INTO cup_goalscorers 
                   (gameweek, fixture_id, team_id, manager_id, points, season_year) 
                   VALUES ('$gameweek', '{$fixture['id']}', '$winning_team_id', 
                           '$top_manager', '$top_points', 
                           '" . $db->real_escape_string($season) . "')");
    }
}


/**
 * Enhanced group standings calculation using your goal rules
 */
function computeCupGroupStandings($group_id, $season = '2025')
{
    global $db;

    // Get teams in group
    $teams_res = $db->query("SELECT cgt.team_id, t.team_name 
                            FROM cup_group_teams cgt 
                            JOIN teams t ON cgt.team_id = t.id 
                            WHERE cgt.group_id = $group_id");

    $standings = [];
    while ($team = $teams_res->fetch_assoc()) {
        $standings[$team['team_id']] = [
            'team_id' => $team['team_id'],
            'team_name' => $team['team_name'],
            'played' => 0,
            'won' => 0,
            'drawn' => 0,
            'lost' => 0,
            'gf' => 0,
            'ga' => 0,
            'gd' => 0,
            'points' => 0
        ];
    }

    // Get all fixtures for this group
    $teamIds = array_keys($standings);
    $teamIdsStr = implode(',', $teamIds);

    $fixtures = $db->query("SELECT * FROM cup_fixtures 
                           WHERE stage = 'Group' 
                           AND season_year = '" . $db->real_escape_string($season) . "' 
                           AND home_team_id IN ($teamIdsStr) 
                           AND away_team_id IN ($teamIdsStr) 
                           AND status = 'Closed'");

    while ($fx = $fixtures->fetch_assoc()) {
        $home = $fx['home_team_id'];
        $away = $fx['away_team_id'];
        $hg = intval($fx['home_goals']);
        $ag = intval($fx['away_goals']);

        $standings[$home]['played']++;
        $standings[$away]['played']++;
        $standings[$home]['gf'] += $hg;
        $standings[$home]['ga'] += $ag;
        $standings[$away]['gf'] += $ag;
        $standings[$away]['ga'] += $ag;

        if ($hg > $ag) {
            $standings[$home]['won']++;
            $standings[$home]['points'] += 3;
            $standings[$away]['lost']++;
        } elseif ($ag > $hg) {
            $standings[$away]['won']++;
            $standings[$away]['points'] += 3;
            $standings[$home]['lost']++;
        } else {
            $standings[$home]['drawn']++;
            $standings[$away]['drawn']++;
            $standings[$home]['points'] += 1;
            $standings[$away]['points'] += 1;
        }
    }

    // Calculate GD and sort (using your existing sorting logic)
    foreach ($standings as &$team) {
        $team['gd'] = $team['gf'] - $team['ga'];
    }

    uasort($standings, function ($a, $b) {
        if ($a['points'] !== $b['points']) return $b['points'] - $a['points'];
        if ($a['gd'] !== $b['gd']) return $b['gd'] - $a['gd'];
        if ($a['gf'] !== $b['gf']) return $b['gf'] - $a['gf'];
        return 0;
    });

    // Add positions
    $position = 1;
    foreach ($standings as &$team) {
        $team['position'] = $position;
        $position++;
    }

    return $standings;
}

/**
 * Update knockout results with aggregate scoring
 */
function updateCupKnockoutResults($gameweek, $season = '2025')
{
    global $db;

    $fixtures = $db->query("SELECT * FROM cup_fixtures 
                           WHERE gameweek = $gameweek 
                           AND season_year = '" . $db->real_escape_string($season) . "' 
                           AND stage != 'Group' 
                           AND status = 'Pending'");

    while ($fixture = $fixtures->fetch_assoc()) {
        $home_points = getCupTeamPoints($fixture['home_team_id'], $gameweek);
        $away_points = getCupTeamPoints($fixture['away_team_id'], $gameweek);

        $goals = applyGoalRules($home_points, $away_points);

        $db->query("UPDATE cup_fixtures 
                   SET home_goals = {$goals['home_goals']}, 
                       away_goals = {$goals['away_goals']}, 
                       status = 'Closed' 
                   WHERE id = {$fixture['id']}");

        // Check if this completes a two-legged tie
        if ($fixture['leg'] == 2) {
            determineCupKnockoutWinner($fixture['tie_identifier'], $season);
        }
    }
}


/**
 * Determine knockout winner after both legs
 */
function determineCupKnockoutWinner($tie_identifier, $season = '2025')
{
    global $db;

    // Get both legs
    $legs = $db->query("SELECT * FROM cup_fixtures 
                       WHERE tie_identifier = '" . $db->real_escape_string($tie_identifier) . "' 
                       AND season_year = '" . $db->real_escape_string($season) . "' 
                       ORDER BY leg");

    $leg1 = $legs->fetch_assoc();
    $leg2 = $legs->fetch_assoc();

    if (!$leg1 || !$leg2) return;

    // Calculate aggregate using goals (not points)
    $agg_home = $leg1['home_goals'] + $leg2['away_goals'];
    $agg_away = $leg1['away_goals'] + $leg2['home_goals'];

    $winner_id = null;

    if ($agg_home > $agg_away) {
        $winner_id = $leg1['home_team_id'];
    } elseif ($agg_away > $agg_home) {
        $winner_id = $leg1['away_team_id'];
    } else {
        // Away goals rule
        $away_goals_home = $leg2['away_goals'];
        $away_goals_away = $leg1['away_goals'];

        if ($away_goals_home > $away_goals_away) {
            $winner_id = $leg1['home_team_id'];
        } elseif ($away_goals_away > $away_goals_home) {
            $winner_id = $leg1['away_team_id'];
        } else {
            // Use random for tiebreaker (or implement penalties)
            $winner_id = mt_rand(0, 1) ? $leg1['home_team_id'] : $leg1['away_team_id'];
        }
    }

    // Update winner in both legs
    $db->query("UPDATE cup_fixtures SET winner_team_id = $winner_id 
               WHERE tie_identifier = '" . $db->real_escape_string($tie_identifier) . "'");

    // Advance to next round
    advanceCupKnockoutRound($leg1['stage'], $winner_id, $season);
}

/**
 * Advance winner to next knockout round
 */
// function advanceCupKnockoutRound($current_round, $winner_id, $season = '2025')
// {
//     global $db;

//     $next_round_map = [
//         'R16' => 'QF',
//         'QF' => 'SF',
//         'SF' => 'F'
//     ];

//     if (!isset($next_round_map[$current_round])) return;

//     $next_round = $next_round_map[$current_round];

//     // Find empty slot in next round
//     $slot = $db->query("SELECT id, home_team_id, away_team_id FROM cup_fixtures 
//                        WHERE stage = '$next_round' 
//                        AND season_year = '" . $db->real_escape_string($season) . "' 
//                        AND (home_team_id IS NULL OR away_team_id IS NULL) 
//                        LIMIT 1");

//     if ($slot_row = $slot->fetch_assoc()) {
//         if ($slot_row['home_team_id'] === null) {
//             $db->query("UPDATE cup_fixtures SET home_team_id = $winner_id 
//                        WHERE id = {$slot_row['id']}");
//         } else {
//             $db->query("UPDATE cup_fixtures SET away_team_id = $winner_id 
//                        WHERE id = {$slot_row['id']}");
//         }
//     }
// }



// function advanceCupKnockoutRound($current_round, $winner_id, $season = '2025')
// {
//     global $db;
//     $next_round_map = [
//         'R16' => 'QF',
//         'QF' => 'SF', 
//         'SF' => 'F'
//     ];

//     if (!isset($next_round_map[$current_round])) return;

//     $next_round = $next_round_map[$current_round];

//     // Check if next round exists, if not create it
//     $round_exists = $db->query("SELECT COUNT(*) as count FROM cup_fixtures 
//                                WHERE stage = '$next_round' 
//                                AND season_year = '" . $db->real_escape_string($season) . "'");
    
//     $round_count = $round_exists->fetch_assoc()['count'];
    
//     if ($round_count == 0) {
//         // Create the next round fixtures
//         createNextRoundFixtures($current_round, $next_round, $season);
//     }

//     // Find empty slot in next round
//     $slot = $db->query("SELECT id, home_team_id, away_team_id FROM cup_fixtures 
//                        WHERE stage = '$next_round' 
//                        AND season_year = '" . $db->real_escape_string($season) . "' 
//                        AND (home_team_id IS NULL OR away_team_id IS NULL) 
//                        ORDER BY tie_identifier, leg 
//                        LIMIT 1");

//     if ($slot_row = $slot->fetch_assoc()) {
//         if ($slot_row['home_team_id'] === null) {
//             $db->query("UPDATE cup_fixtures SET home_team_id = $winner_id 
//                        WHERE id = {$slot_row['id']}");
//         } else {
//             $db->query("UPDATE cup_fixtures SET away_team_id = $winner_id 
//                        WHERE id = {$slot_row['id']}");
//         }
//         return true;
//     }
    
//     return false;
// }


function advanceCupKnockoutRound($current_round, $winner_id, $season = '2025')
{
    global $db;

    $next_round_map = [
        'R16' => 'QF',
        'QF' => 'SF',
        'SF' => 'F'
    ];

    if (!isset($next_round_map[$current_round])) return false;

    $next_round = $next_round_map[$current_round];

    // Check if next round exists, if not create it
    $round_exists = $db->query("SELECT COUNT(*) as count FROM cup_fixtures 
                               WHERE stage = '$next_round' 
                               AND season_year = '" . $db->real_escape_string($season) . "'");
    
    $round_count = $round_exists->fetch_assoc()['count'];
    
    if ($round_count == 0) {
        createNextRoundFixtures($current_round, $next_round, $season);
    }

    // Store this winner in a temporary pairing table
    $db->query("INSERT INTO cup_round_winners (round, season_year, team_id, processed) 
               VALUES ('$next_round', '" . $db->real_escape_string($season) . "', $winner_id, 0)");
    
    // Check if we have pairs to process
    $pairs = $db->query("SELECT w1.team_id as team1, w2.team_id as team2, 
                         w1.id as id1, w2.id as id2
                         FROM cup_round_winners w1
                         JOIN cup_round_winners w2 ON w1.round = w2.round 
                         AND w1.season_year = w2.season_year
                         WHERE w1.round = '$next_round' 
                         AND w1.season_year = '" . $db->real_escape_string($season) . "'
                         AND w1.processed = 0 AND w2.processed = 0
                         AND w1.id < w2.id
                         LIMIT 1");
    
    if ($pair = $pairs->fetch_assoc()) {
        // We have a pair - find an empty tie
        $empty_tie = $db->query("SELECT tie_identifier 
                                FROM cup_fixtures 
                                WHERE stage = '$next_round' 
                                AND season_year = '" . $db->real_escape_string($season) . "'
                                GROUP BY tie_identifier
                                HAVING SUM(CASE WHEN home_team_id IS NOT NULL THEN 1 ELSE 0 END) = 0
                                AND SUM(CASE WHEN away_team_id IS NOT NULL THEN 1 ELSE 0 END) = 0
                                LIMIT 1");
        
        if ($tie_row = $empty_tie->fetch_assoc()) {
            $tie_identifier = $tie_row['tie_identifier'];
            
            // Assign team1 as home (leg1) / away (leg2)
            $db->query("UPDATE cup_fixtures SET home_team_id = {$pair['team1']} 
                       WHERE stage = '$next_round' 
                       AND season_year = '" . $db->real_escape_string($season) . "' 
                       AND tie_identifier = '$tie_identifier' 
                       AND leg = 1");
            
            $db->query("UPDATE cup_fixtures SET away_team_id = {$pair['team1']} 
                       WHERE stage = '$next_round' 
                       AND season_year = '" . $db->real_escape_string($season) . "' 
                       AND tie_identifier = '$tie_identifier' 
                       AND leg = 2");
            
            // Assign team2 as away (leg1) / home (leg2)  
            $db->query("UPDATE cup_fixtures SET away_team_id = {$pair['team2']} 
                       WHERE stage = '$next_round' 
                       AND season_year = '" . $db->real_escape_string($season) . "' 
                       AND tie_identifier = '$tie_identifier' 
                       AND leg = 1");
            
            $db->query("UPDATE cup_fixtures SET home_team_id = {$pair['team2']} 
                       WHERE stage = '$next_round' 
                       AND season_year = '" . $db->real_escape_string($season) . "' 
                       AND tie_identifier = '$tie_identifier' 
                       AND leg = 2");
            
            // Mark both as processed
            $db->query("UPDATE cup_round_winners SET processed = 1 
                       WHERE id IN ({$pair['id1']}, {$pair['id2']})");
            
            return true;
        }
    }
    
    return true; // Winner stored, will be paired later
}

function createNextRoundFixtures($current_round, $next_round, $season)
{
    global $db;
    
    // Determine number of ties and gameweek schedule based on current round
    $round_config = [
        'R16' => ['next_ties' => 4, 'start_gw' => 28, 'interval' => 2],  // QF after R16
        'QF' => ['next_ties' => 2, 'start_gw' => 32, 'interval' => 2],   // SF after QF  
        'SF' => ['next_ties' => 1, 'start_gw' => 38, 'interval' => 0]    // Final after SF (single leg)
    ];
    
    if (!isset($round_config[$current_round])) return;
    
    $config = $round_config[$current_round];
    $num_ties = $config['next_ties'];
    $start_gw = $config['start_gw'];
    $interval = $config['interval'];
    
    for ($tieIndex = 1; $tieIndex <= $num_ties; $tieIndex++) {
        $tieIdentifier = $next_round . '-' . $tieIndex;
        
        if ($next_round == 'F') {
            // Final is single leg
            $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) 
                       VALUES ('$next_round', '" . $db->real_escape_string($season) . "', $start_gw, NULL, NULL, 1, '$tieIdentifier', 'Pending')");
        } else {
            // Two-legged ties
            // Leg 1
            $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) 
                       VALUES ('$next_round', '" . $db->real_escape_string($season) . "', $start_gw, NULL, NULL, 1, '$tieIdentifier', 'Pending')");
            // Leg 2  
            $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) 
                       VALUES ('$next_round', '" . $db->real_escape_string($season) . "', " . ($start_gw + $interval) . ", NULL, NULL, 2, '$tieIdentifier', 'Pending')");
        }
        
        // $start_gw += ($interval * 2) + 1; // Space between ties
    }
    
    return $num_ties;
}





/**
 * Main integration point - call this from your existing system
 */
// function integrateChampionsCup($current_gw, $season = '2025')
// {
//     try {
//         runEnhancedChampionsCupWorkflow($current_gw, $season);
//         return ["success" => true, "message" => "Champions Cup updated for GW$current_gw"];
//     } catch (Exception $e) {
//         error_log("Champions Cup Error: " . $e->getMessage());
//         return ["success" => false, "message" => $e->getMessage()];
//     }
// }
