<?php
// set_time_limit(600);
include('../connect.php');

function getOverallTeamTotals($div, $upToGw) {
    global $db;
    // Returns associative: team_id => total_net_points
    $totals = [];

    // Efficient single query: join teams -> manager ids -> manager_history
    // But simplest: fetch teams and their managers then sum via manager_history grouping.
    $teamsRes = $db->query("SELECT id, manager1, manager2, manager3, manager4, manager5 FROM teams WHERE division = '$div'");
    $teams = [];
    while ($t = mysqli_fetch_assoc($teamsRes)) {
        $teams[$t['id']] = [
            'managers' => array_filter([$t['manager1'],$t['manager2'],$t['manager3'],$t['manager4'],$t['manager5']])
        ];
        $totals[$t['id']] = 0;
    }

    // Pull all manager_history up to $upToGw once
    $mhRes = $db->query("SELECT manager_id, event, net_points FROM manager_history WHERE event <= " . intval($upToGw));
    $mgrPoints = [];
    while ($r = mysqli_fetch_assoc($mhRes)) {
        $mgrPoints[$r['manager_id']][$r['event']] = intval($r['net_points']);
    }

    // Sum per team
    foreach ($teams as $teamId => $t) {
        $sum = 0;
        foreach ($t['managers'] as $m) {
            if (isset($mgrPoints[$m])) {
                foreach ($mgrPoints[$m] as $gw => $p) $sum += $p;
            }
        }
        $totals[$teamId] = $sum;
    }

    return $totals; // team_id => points
}


function generateCupQualifiers($season='2025') {
    global $db;
    $upToGw = 10;
    $totals = getOverallTeamTotals(1, $upToGw);

    // defending champion id — fetch from your history table
    $defenderRes = $db->query("SELECT winner_team_id FROM cup_history ORDER BY id DESC LIMIT 1");
    $defender = mysqli_fetch_assoc($defenderRes)['winner_team_id'] ?? null;
    $defenderDiv = getTeamDivision($defender);

    // clear previous cup_teams for season (if re-run)
    $db->query("DELETE FROM cup_teams WHERE season_year='". $db->real_escape_string($season) ."'");

    // Insert defending champion
    if ($defender) {
        $db->query("INSERT INTO cup_teams (team_id, division, qualifying_rank, qualified_at_gw, seed, season_year) VALUES ($defender, $defenderDiv, 0, 10, 1, '".$db->real_escape_string($season)."')");
    }

    // Now pick top teams per division
    // assume divisions stored in teams.division (1,2,3)
    $limits = [1 => 12, 2 => 10, 3 => 9];
    foreach ($limits as $div => $limit) {
        // get all teams in division and their totals
        $teamRes = $db->query("SELECT id FROM teams WHERE division = $div");
        $divTeams = [];
        while ($r = mysqli_fetch_assoc($teamRes)) $divTeams[] = $r['id'];

        // build sortable array of (team => points)
        $arr = [];
        foreach ($divTeams as $tid) {
            $arr[$tid] = $totals[$tid] ?? 0;
        }
        arsort($arr); // desc by points

        $rank = 1;
        foreach ($arr as $tid => $pts) {
            if ($rank > $limit) break;
            $seed = $rank + ($div==1?0:($div==2?12:22)); // optional seeding offset
            $db->query("INSERT INTO cup_teams (team_id, division, qualifying_rank, qualified_at_gw, seed, season_year) VALUES ($tid, $div, $rank, 10, $seed, '".$db->real_escape_string($season)."')");
            $rank++;
        }
    }
}



function createCupGroups($season='2025') {
    global $db;
    // read qualified teams and their seed
    $q = $db->query("SELECT team_id, seed FROM cup_teams WHERE season_year='". $db->real_escape_string($season)."' ORDER BY seed ASC");
    $teams = [];
    while ($r = mysqli_fetch_assoc($q)) $teams[] = $r;

    if (count($teams) !== 32) {
        throw new Exception("Qualification not 32 teams: " . count($teams));
    }

    // Build 4 pots by seed order (1..8,9..16 etc.)
    $pots = array_chunk($teams, 8); // pots[0]=top8, pots[1]=next8, etc.

    // clear groups for season
    $db->query("DELETE FROM cup_groups WHERE season_year='".$db->real_escape_string($season)."'");
    $db->query("DELETE FROM cup_group_teams WHERE id IN (SELECT cgt.id FROM cup_group_teams cgt JOIN cup_groups cg ON cgt.group_id = cg.id WHERE cg.season_year = '".$db->real_escape_string($season)."')");

    $groupNames = range('A','H');
    // create empty groups
    foreach ($groupNames as $g) {
        $db->query("INSERT INTO cup_groups (group_name, season_year) VALUES ('{$g}', '".$db->real_escape_string($season)."')");
        $gid = $db->insert_id;
        // For each pot pick a team at random without duplication
        for ($pot=0;$pot<4;$pot++) {
            // pick a random entry from pots[$pot]
            $idx = array_rand($pots[$pot]);
            $team = $pots[$pot][$idx];
            // remove chosen
            array_splice($pots[$pot], $idx, 1);
            // record in group_team with seed 1..4
            $db->query("INSERT INTO cup_group_teams (group_id, team_id, group_seed) VALUES ($gid, {$team['team_id']}, ".($pot+1).")");
        }
    }
}


function generateGroupFixtures($season='2025', $start_gw = 12, $interval = 2) {
    global $db;
    // get groups and members
    $groups = $db->query("SELECT id, group_name FROM cup_groups WHERE season_year='". $db->real_escape_string($season)."'");
    $gw = intval($start_gw);

    // optional: clear existing fixtures for season/stage
    $db->query("DELETE FROM cup_fixtures WHERE season_year='".$db->real_escape_string($season)."' AND stage='Group'");

    while ($g = mysqli_fetch_assoc($groups)) {
        // fetch teams in group ordered by group_seed
        $res = $db->query("SELECT team_id FROM cup_group_teams WHERE group_id={$g['id']} ORDER BY group_seed ASC");
        $teams = [];
        while ($t = mysqli_fetch_assoc($res)) $teams[] = $t['team_id'];

        // create pairings: for i < j both legs (home/away)
        $pairs = [];
        for ($i=0;$i<4;$i++) {
            for ($j=$i+1;$j<4;$j++) {
                $pairs[] = [$teams[$i], $teams[$j]];
                $pairs[] = [$teams[$j], $teams[$i]];
            }
        }

        // optional: shuffle order of pairs so schedule not predictable
        shuffle($pairs);

        foreach ($pairs as $p) {
            $home = $p[0]; $away = $p[1];
            $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status)VALUES ('Group', '".$db->real_escape_string($season)."', $gw, $home, $away, 1, NULL, 'Pending')");
            $gw += $interval;
        }
    }
}



function checkAndGenerateKnockouts($season='2025') {
    global $db;
    // Are all Group stage fixtures Closed?
    $left = $db->query("SELECT COUNT(*) AS cnt FROM cup_fixtures WHERE season_year='".$db->real_escape_string($season)."' AND stage='Group' AND status != 'Closed'")->fetch_assoc()['cnt'];
    if ($left > 0) return false; // still ongoing

    // compute group standings
    $groups = $db->query("SELECT id FROM cup_groups WHERE season_year='".$db->real_escape_string($season)."'");
    $r16_teams = [];

    while ($g = $groups->fetch_assoc()) {
        $gid = $g['id'];
        // initialize stats
        $stats = []; // team_id => [points, gf, ga, gd]
        $res = $db->query("SELECT team_id FROM cup_group_teams WHERE group_id=$gid");
        while ($t = $res->fetch_assoc()) $stats[$t['team_id']] = ['points'=>0,'gf'=>0,'ga'=>0];

        // get fixtures for this group
        $fx = $db->query("SELECT * FROM cup_fixtures WHERE stage='Group' AND season_year='". $db->real_escape_string($season)."' AND (home_team_id IN (SELECT team_id FROM cup_group_teams WHERE group_id=$gid) AND away_team_id IN (SELECT team_id FROM cup_group_teams WHERE group_id=$gid))");
        while ($f = $fx->fetch_assoc()) {
            if ($f['home_goals'] === null || $f['away_goals'] === null) continue;
            $home = $f['home_team_id']; $away = $f['away_team_id'];
            $hg = intval($f['home_goals']); $ag = intval($f['away_goals']);
            $stats[$home]['gf'] += $hg; $stats[$home]['ga'] += $ag;
            $stats[$away]['gf'] += $ag; $stats[$away]['ga'] += $hg;
            if ($hg > $ag) {
                $stats[$home]['points'] += 3;
            } elseif ($hg < $ag) {
                $stats[$away]['points'] += 3;
            } else {
                $stats[$home]['points'] += 1;
                $stats[$away]['points'] += 1;
            }
        }

        // compute GD and sort
        foreach ($stats as $tid => &$s) $s['gd'] = $s['gf'] - $s['ga'];
        uasort($stats, function($a,$b){
            if ($a['points'] !== $b['points']) return $b['points'] - $a['points'];
            if ($a['gd'] !== $b['gd']) return $b['gd'] - $a['gd'];
            if ($a['gf'] !== $b['gf']) return $b['gf'] - $a['gf'];
            return 0;
        });

        // pick top 2
        $group_order = array_keys($stats);
        $top2 = array_slice($group_order,0,2);
        $r16_teams = array_merge($r16_teams, $top2);
    }

    // Now create Round of 16 pairings (simple bracket: A1 vs B2 etc.)
    // Example mapping (A1 v B2), (C1 v D2), (E1 v F2), (G1 v H2), (B1 v A2), (D1 v C2), (F1 v E2), (H1 v G2)
    // We need to map groups to their A1, B2 etc. Build map:
    $groupTeams = [];
    $grRes = $db->query("SELECT id, group_name FROM cup_groups WHERE season_year='".$db->real_escape_string($season)."'");
    while ($g = $grRes->fetch_assoc()) {
        // obtain ordered teams (same code as above can be refactored into a function)
        // For brevity assume $orderedTeams = computeOrderedTeamsForGroup($g['id']);
        $orderedTeams = computeOrderedTeamsForGroup($db, $g['id'], $season); // you implement
        $groupTeams[$g['group_name']] = $orderedTeams; // array of team_ids, 0-based [0]=1st
    }

    // create R16 pairings (example)
    $r16_matches = [
      [$groupTeams['A'][0], $groupTeams['B'][1]],
      [$groupTeams['C'][0], $groupTeams['D'][1]],
      [$groupTeams['E'][0], $groupTeams['F'][1]],
      [$groupTeams['G'][0], $groupTeams['H'][1]],
      [$groupTeams['B'][0], $groupTeams['A'][1]],
      [$groupTeams['D'][0], $groupTeams['C'][1]],
      [$groupTeams['F'][0], $groupTeams['E'][1]],
      [$groupTeams['H'][0], $groupTeams['G'][1]]
    ];

    // schedule R16 fixtures: two legs. Choose GWs: e.g. start at GW( last group gw + 2) and every 2 gws
    $start_gw = determineR16StartGW($db, $season);
    $interval = 2;
    $tieIndex = 1;
    foreach ($r16_matches as $pair) {
        // leg 1
        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) VALUES ('R16','".$db->real_escape_string($season)."',$start_gw, {$pair[0]}, {$pair[1]}, 1, 'R16-{$tieIndex}', 'Pending')");
        // leg 2
        $db->query("INSERT INTO cup_fixtures (stage, season_year, gameweek, home_team_id, away_team_id, leg, tie_identifier, status) VALUES ('R16','".$db->real_escape_string($season)."',".($start_gw+$interval).", {$pair[1]}, {$pair[0]}, 2, 'R16-{$tieIndex}', 'Pending')");
        $start_gw += ($interval * 2); // space between ties if desired
        $tieIndex++;
    }

    return true;
}
